# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

A cross-platform video combiner tool for merging multiple MP4 files from NVR (Network Video Recorder) systems using ffmpeg's concat demuxer. All implementations concatenate videos without re-encoding (using `-c copy`), making the process fast and lossless.

## Available Implementations

This repository provides implementations in multiple languages for broad platform compatibility:

1. **PowerShell** (`Join-Videos.ps1`) - Windows native
2. **Python** (`join_videos.py`) - Cross-platform, requires Python 3.6+
3. **Bash** (`join_videos.sh`) - Linux/Mac/WSL
4. **Go** (`join_videos.go`) - Compiled, cross-platform binary
5. **Rust** (`join_videos.rs` + `Cargo.toml`) - Compiled, cross-platform binary
6. **Node.js** (`join_videos.js` + `package.json`) - Cross-platform, requires Node.js 12+

All implementations share the same core functionality and command-line interface design (where applicable).

## Running the Scripts

All implementations now share consistent interactive behavior:
- Prompts for video directory if not provided
- Auto-detects ffmpeg with deep search
- Offers to install ffmpeg if not found

### PowerShell (Windows)
```powershell
.\Join-Videos.ps1                              # Interactive mode
.\Join-Videos.ps1 C:\Videos                    # With directory
.\Join-Videos.ps1 -Output out.mp4 C:\Videos    # With options
```

### Python (All Platforms)
```bash
python join_videos.py                          # Interactive mode
python join_videos.py /path/to/videos          # With directory
python join_videos.py /path/to/videos -o out.mp4
```

### Bash (Linux/Mac/WSL)
```bash
chmod +x join_videos.sh
./join_videos.sh                               # Interactive mode
./join_videos.sh /path/to/videos -o output.mp4
```

### Go (Compile & Run)
```bash
go build join_videos.go
./join_videos                                  # Interactive mode
./join_videos /path/to/videos -output output.mp4
```

### Rust (Compile & Run)
```bash
cargo build --release
./target/release/join_videos                   # Interactive mode
./target/release/join_videos /path/to/videos -o output.mp4
```

### Node.js (All Platforms)
```bash
node join_videos.js                            # Interactive mode
node join_videos.js /path/to/videos -o output.mp4
```

## Common Command-Line Options

All implementations support:
- `VIDEO_DIR` - Positional argument for video directory
- `-o, --output FILE` - Set output filename (default: Combined_Output.mp4)
- `--ffmpeg PATH` - Custom ffmpeg path (auto-detected by default)
- `--keep-concat` - Preserve concat.txt after completion
- `-h, --help` - Show help message

PowerShell uses `-Output`, `-FFmpeg`, `-KeepConcat`, `-Help` syntax.

## How It Works

All implementations follow the same workflow:

1. Prompt for video directory if not provided (with platform-specific default)
2. Search for ffmpeg in PATH and common locations; prompt if not found
3. Find all `.mp4` files in the specified directory
4. Sort files alphabetically by filename
5. Generate a `concat.txt` file with escaped file paths in ffmpeg concat format
6. Execute ffmpeg with the concat demuxer to combine videos without re-encoding
7. Clean up temporary `concat.txt` file (unless `--keep-concat` is specified)

## Prerequisites

**All implementations require:**
- ffmpeg installed and accessible (scripts auto-detect and offer to install if not found)
- Source videos must have compatible formats/codecs for concat to work
- Sufficient disk space for the output file

**Language-specific:**
- Python: Python 3.6+
- Node.js: Node.js 12.0.0+
- Go: Go 1.16+ (for building)
- Rust: Rust 1.56+ (for building)
- Bash: Bash 4.0+

## Key Technical Details

- Uses ffmpeg's concat demuxer (`-f concat`) with `-safe 0` flag for absolute paths
- The `-c copy` flag ensures no re-encoding (stream copy only - fast and lossless)
- Path escaping handles single quotes in filenames: `'` → `'\''`
- Videos are concatenated in alphabetical order by filename
- All implementations create the same `concat.txt` format for ffmpeg

## Choosing an Implementation

- **Windows users**: PowerShell (no dependencies) or Python (more flexible)
- **Linux/Mac users**: Bash (native) or Python (cross-platform)
- **Performance-critical**: Use compiled binaries (Go or Rust)
- **Maximum portability**: Python or Node.js (widely available runtimes)
- **No runtime dependencies**: Compile Go or Rust to standalone binary
