###############################################################################
# NVR Video Combiner - PowerShell Version
# Combines multiple MP4 video files using ffmpeg's concat demuxer (no re-encoding)
###############################################################################

param(
    [string]$VideoDir,
    [Alias('o')]
    [string]$Output = "Combined_Output.mp4",
    [string]$FFmpeg,
    [switch]$KeepConcat,
    [Alias('h')]
    [switch]$Help
)

# Default values
$DefaultVideoDir = "$env:USERPROFILE\Videos\NVR"

# Colors for output
function Write-Green { param($msg) Write-Host $msg -ForegroundColor Green }
function Write-Yellow { param($msg) Write-Host $msg -ForegroundColor Yellow }
function Write-Red { param($msg) Write-Host $msg -ForegroundColor Red }

# Usage information
function Show-Usage {
    Write-Host "NVR Video Combiner - PowerShell Version`n"
    Write-Host "Usage: .\Join-Videos.ps1 [OPTIONS] [VIDEO_DIR]`n"
    Write-Host "Combines multiple MP4 video files using ffmpeg (no re-encoding)`n"
    Write-Host "Arguments:"
    Write-Host "    VIDEO_DIR           Directory containing MP4 files (default: $DefaultVideoDir)`n"
    Write-Host "Options:"
    Write-Host "    -Output FILE        Output filename (default: Combined_Output.mp4)"
    Write-Host "    -FFmpeg PATH        Path to ffmpeg executable (auto-detected if not specified)"
    Write-Host "    -KeepConcat         Keep the concat.txt file after completion"
    Write-Host "    -Help               Show this help message`n"
    Write-Host "Examples:"
    Write-Host "    .\Join-Videos.ps1 C:\Videos"
    Write-Host "    .\Join-Videos.ps1 -Output MyVideo.mp4 C:\Videos"
    Write-Host "    .\Join-Videos.ps1 -FFmpeg C:\ffmpeg\ffmpeg.exe C:\Videos"
    exit 0
}

# Find ffmpeg with comprehensive search
function Find-FFmpeg {
    Write-Yellow "Searching for ffmpeg..."

    # Check PATH first
    $inPath = Get-Command ffmpeg -ErrorAction SilentlyContinue
    if ($inPath) {
        return $inPath.Source
    }

    # Deep search in common locations
    $searchPaths = @(
        "C:\ffmpeg",
        "C:\Program Files\ffmpeg",
        "C:\Program Files (x86)\ffmpeg",
        "$env:LOCALAPPDATA\Microsoft\WinGet\Packages",
        "$env:USERPROFILE\scoop\apps\ffmpeg",
        "$env:ChocolateyInstall\bin",
        "C:\tools"
    )

    Write-Yellow "Performing deep search in common directories..."
    foreach ($base in $searchPaths) {
        if (Test-Path $base) {
            $found = Get-ChildItem -Path $base -Filter "ffmpeg.exe" -Recurse -ErrorAction SilentlyContinue | Select-Object -First 1
            if ($found) {
                return $found.FullName
            }
        }
    }

    return $null
}

# Show help if requested
if ($Help) {
    Show-Usage
}

# Handle positional argument for VideoDir
if (-not $VideoDir -and $args.Count -gt 0) {
    $VideoDir = $args[0]
}

# Set default video directory if not specified - prompt user
if (-not $VideoDir) {
    $userInput = Read-Host "Enter video directory path [$DefaultVideoDir]"
    if ([string]::IsNullOrWhiteSpace($userInput)) {
        $VideoDir = $DefaultVideoDir
    } else {
        $VideoDir = $userInput
    }
}

# Validate video directory
if (-not (Test-Path $VideoDir -PathType Container)) {
    Write-Red "Error: Directory does not exist: $VideoDir"
    exit 1
}

# Find ffmpeg with comprehensive search and user prompts
if (-not $FFmpeg) {
    $FFmpeg = Find-FFmpeg
    if ($FFmpeg) {
        Write-Green "Found ffmpeg: $FFmpeg"
    } else {
        Write-Red "ffmpeg not found after comprehensive search"
        $userPath = Read-Host "Enter path to ffmpeg executable (or press Enter to open download page)"

        if (-not [string]::IsNullOrWhiteSpace($userPath)) {
            if (Test-Path $userPath -PathType Leaf) {
                $FFmpeg = $userPath
                Write-Green "Using: $FFmpeg"
            } else {
                Write-Red "Error: Not a valid executable: $userPath"
                exit 1
            }
        } else {
            $install = Read-Host "Open ffmpeg download page? (y/n)"
            if ($install -match '^[Yy]') {
                # Try winget first
                $hasWinget = Get-Command winget -ErrorAction SilentlyContinue
                if ($hasWinget) {
                    $useWinget = Read-Host "Install via winget? (y/n)"
                    if ($useWinget -match '^[Yy]') {
                        Write-Yellow "Installing ffmpeg via winget..."
                        winget install ffmpeg
                        $FFmpeg = Find-FFmpeg
                        if (-not $FFmpeg) {
                            Write-Red "Error: Installation may have succeeded but ffmpeg not found. Please restart PowerShell."
                            exit 1
                        }
                    }
                }

                if (-not $FFmpeg) {
                    Start-Process "https://ffmpeg.org/download.html"
                    Write-Host "Please download and install ffmpeg, then run this script again."
                    exit 1
                }
            } else {
                Write-Red "ffmpeg required to continue"
                exit 1
            }
        }
    }
}

if (-not (Test-Path $FFmpeg -PathType Leaf)) {
    Write-Red "Error: ffmpeg not found at: $FFmpeg"
    exit 1
}

# Setup paths
$concatFile = Join-Path $VideoDir "concat.txt"
$outputPath = Join-Path $VideoDir $Output

# Find all MP4 files
$videoFiles = Get-ChildItem -Path $VideoDir -Filter "*.mp4" -File | Sort-Object Name

if ($videoFiles.Count -eq 0) {
    Write-Red "Error: No .mp4 files found in $VideoDir"
    exit 1
}

# Display found videos
Write-Green "Found $($videoFiles.Count) video file(s):"
foreach ($video in $videoFiles) {
    Write-Host "  - $($video.Name)"
}

# Create concat file
Write-Host ""
Write-Yellow "Creating concat file: $concatFile"
$videoFiles | ForEach-Object {
    $escapedPath = $_.FullName.Replace("'", "'\\''")
    "file '$escapedPath'"
} | Set-Content -Path $concatFile -Encoding UTF8

# Run ffmpeg
Write-Host ""
Write-Yellow "Running ffmpeg..."
Write-Host "Output will be: " -NoNewline
Write-Green $outputPath
Write-Host ""

$success = $false
try {
    & $FFmpeg -f concat -safe 0 -i $concatFile -c copy $outputPath
    if ($LASTEXITCODE -eq 0) {
        Write-Host ""
        Write-Green "✓ Successfully created: $outputPath"
        $success = $true
    } else {
        Write-Host ""
        Write-Red "✗ Error running ffmpeg"
    }
} catch {
    Write-Host ""
    Write-Red "✗ Error running ffmpeg: $_"
}

# Cleanup concat file
if (-not $KeepConcat -and (Test-Path $concatFile)) {
    Remove-Item $concatFile
    Write-Host "Cleaned up: $concatFile"
}

if ($success) { exit 0 } else { exit 1 }
