# NVR Video Combiner

A cross-platform utility for combining multiple MP4 video files from NVR (Network Video Recorder) systems using ffmpeg's concat demuxer. Fast and lossless - no re-encoding required!

## Features

- **Cross-platform**: Implementations available for Windows, macOS, and Linux
- **Multiple languages**: Choose from PowerShell, Python, Bash, Go, Rust, or Node.js
- **Fast**: Uses ffmpeg's concat demuxer with stream copy (no re-encoding)
- **Lossless**: Preserves original video quality
- **Smart ffmpeg detection**: Deep search across common installation directories
- **Interactive prompts**: Guides you through directory and ffmpeg setup
- **Auto-install**: Offers to install ffmpeg via package managers if not found
- **Flexible**: Command-line options for customization
- **Alphabetical ordering**: Combines videos in sorted filename order

## Quick Start

### Prerequisites

- **ffmpeg** - Will be auto-detected or you'll be prompted to install it
  - The scripts search common locations automatically
  - If not found, you can provide a custom path or install via package manager

### Choose Your Implementation

Pick the version that best suits your environment:

| Implementation | Best For | Runtime Required |
|---------------|----------|------------------|
| **PowerShell** | Windows users | None (built-in) |
| **Python** | Cross-platform, flexibility | Python 3.6+ |
| **Bash** | Linux/Mac/WSL native | Bash 4.0+ |
| **Go** | Compiled binary, performance | Go 1.16+ (for building) |
| **Rust** | Compiled binary, maximum performance | Rust 1.56+ (for building) |
| **Node.js** | Cross-platform, modern | Node.js 12+ |

## Usage Examples

All implementations share the same interactive behavior:
1. If no directory specified, prompts for one (with sensible default)
2. Automatically searches for ffmpeg in common locations
3. If ffmpeg not found, offers to provide path or install it

### PowerShell (Windows)

```powershell
# Interactive mode - prompts for directory
.\Join-Videos.ps1

# With directory argument
.\Join-Videos.ps1 C:\Videos\NVR

# With all options
.\Join-Videos.ps1 -Output MyVideo.mp4 C:\Videos\NVR

# View help
.\Join-Videos.ps1 -Help
```

### Python

```bash
# Interactive mode - prompts for directory
python join_videos.py

# With directory argument
python join_videos.py /path/to/videos

# With all options
python join_videos.py /path/to/videos -o MyOutput.mp4 --keep-concat

# View help
python join_videos.py --help
```

### Bash (Linux/Mac/WSL)

```bash
# Make executable
chmod +x join_videos.sh

# Interactive mode - prompts for directory
./join_videos.sh

# With directory argument
./join_videos.sh /path/to/videos

# With options
./join_videos.sh -o output.mp4 /path/to/videos

# View help
./join_videos.sh --help
```

### Go

```bash
# Build the binary
go build join_videos.go

# Interactive mode
./join_videos

# With directory argument
./join_videos /path/to/videos

# With options
./join_videos -output MyVideo.mp4 /path/to/videos

# Or build and run in one step
go run join_videos.go /path/to/videos
```

### Rust

```bash
# Build optimized binary with Cargo
cargo build --release

# Interactive mode
./target/release/join_videos

# With directory argument
./target/release/join_videos /path/to/videos

# With options
./target/release/join_videos -o output.mp4 /path/to/videos
```

### Node.js

```bash
# Interactive mode
node join_videos.js

# With directory argument
node join_videos.js /path/to/videos

# With options
node join_videos.js -o output.mp4 /path/to/videos

# Or make executable (Unix)
chmod +x join_videos.js
./join_videos.js /path/to/videos
```

## Command-Line Options

All implementations support these options:

| Option | Description |
|--------|-------------|
| `VIDEO_DIR` | Directory containing MP4 files (positional argument) |
| `-o, --output FILE` | Output filename (default: `Combined_Output.mp4`) |
| `--ffmpeg PATH` | Custom path to ffmpeg executable |
| `--keep-concat` | Don't delete concat.txt after completion |
| `-h, --help` | Show help message |

**PowerShell uses slightly different syntax:**
- `-Output` instead of `-o`
- `-FFmpeg` instead of `--ffmpeg`
- `-KeepConcat` instead of `--keep-concat`
- `-Help` instead of `-h`

## Interactive Features

### Directory Prompt

When no directory is provided, you'll be prompted:
```
Enter video directory path [/home/user/Videos/NVR]:
```
Press Enter to use the default or type a custom path.

### ffmpeg Auto-Detection

The scripts search for ffmpeg in:
- System PATH
- Common installation directories:
  - **Windows**: `C:\ffmpeg`, `C:\Program Files\ffmpeg`, WinGet packages, Scoop, Chocolatey
  - **macOS**: `/usr/local/bin`, `/opt/homebrew/bin`, Homebrew paths
  - **Linux**: `/usr/bin`, `/usr/local/bin`, `/opt`, `/snap/bin`, user home directories

### ffmpeg Installation Prompt

If ffmpeg isn't found, you'll see:
```
ffmpeg not found after comprehensive search
Enter path to ffmpeg executable (or press Enter to install):
```

Options:
1. Enter a custom path to ffmpeg
2. Press Enter to install via package manager:
   - **Windows**: winget
   - **macOS**: Homebrew
   - **Linux**: apt-get, yum, or pacman

## How It Works

1. **Directory Selection**: Uses provided path or prompts interactively
2. **ffmpeg Detection**: Searches PATH and common locations, prompts if needed
3. **Discovery**: Finds all `.mp4` files in the specified directory
4. **Sorting**: Sorts files alphabetically by filename
5. **Concat File**: Generates a `concat.txt` with properly escaped file paths
6. **Concatenation**: Runs ffmpeg with the concat demuxer (no re-encoding)
7. **Cleanup**: Removes temporary `concat.txt` (unless `--keep-concat` is used)

### Technical Details

- Uses ffmpeg's **concat demuxer** (`-f concat`)
- **Stream copy mode** (`-c copy`) - no re-encoding
- **Safe mode disabled** (`-safe 0`) to allow absolute paths
- **Path escaping** handles special characters including single quotes
- Videos must have **compatible codecs** for concatenation to work

## File Structure

```
nvr_combine/
├── Join-Videos.ps1      # PowerShell implementation
├── join_videos.py       # Python implementation
├── join_videos.sh       # Bash implementation
├── join_videos.go       # Go implementation
├── join_videos.rs       # Rust implementation
├── Cargo.toml           # Rust build configuration
├── join_videos.js       # Node.js implementation
├── package.json         # Node.js package configuration
├── CLAUDE.md            # Claude Code guidance
└── README.md            # This file
```

## Building Compiled Versions

### Go Binary

```bash
# Build for current platform
go build -o join_videos join_videos.go

# Cross-compile for different platforms
GOOS=windows GOARCH=amd64 go build -o join_videos.exe join_videos.go
GOOS=linux GOARCH=amd64 go build -o join_videos_linux join_videos.go
GOOS=darwin GOARCH=amd64 go build -o join_videos_mac join_videos.go
```

### Rust Binary

```bash
# Build optimized release
cargo build --release

# The binary will be in target/release/join_videos

# Cross-compile (requires additional setup)
cargo build --release --target x86_64-pc-windows-gnu
cargo build --release --target x86_64-unknown-linux-gnu
```

## Troubleshooting

### ffmpeg not found

**Scenario**: ffmpeg isn't detected automatically

**Solutions**:
1. Let the script install it for you (press Enter when prompted)
2. Install manually:
   ```bash
   # macOS
   brew install ffmpeg

   # Ubuntu/Debian
   sudo apt install ffmpeg

   # Windows (using winget)
   winget install ffmpeg
   ```
3. Specify custom path: `--ffmpeg /path/to/ffmpeg`

### No .mp4 files found

**Error**: `No .mp4 files found in directory`

**Solution**:
- Verify the directory path is correct
- Ensure files have `.mp4` extension
- Check file permissions

### Concat fails or produces errors

**Error**: Videos won't concatenate or ffmpeg reports errors

**Solution**:
- Ensure all videos have the **same codec**, resolution, and frame rate
- Videos must be compatible for stream copy to work
- Try re-encoding one or all videos to match formats first

### Permission denied

**Error**: `Permission denied` when running scripts

**Solution**:
```bash
# Make script executable (Unix/Linux/Mac)
chmod +x join_videos.sh
chmod +x join_videos.py
chmod +x join_videos.js
```

## FAQ

**Q: Which implementation should I use?**

A:
- Windows: PowerShell (simplest) or Python (most flexible)
- macOS/Linux: Bash (native) or Python (cross-platform)
- Performance: Go or Rust compiled binaries
- No dependencies: Build Go/Rust to standalone binary

**Q: Will this work with other video formats?**

A: The scripts currently look for `.mp4` files only. You can modify the code to support other formats, but ensure they're compatible with ffmpeg's concat demuxer.

**Q: Why is the output file the same size as the sum of inputs?**

A: Because we use stream copy (`-c copy`), which doesn't re-encode. The output preserves all original video data losslessly.

**Q: Can I customize the sort order?**

A: Yes, modify the sorting logic in the script. Currently files are sorted alphabetically by filename.

**Q: What happens to the concat.txt file?**

A: By default it's deleted after combining. Use `--keep-concat` to preserve it for debugging.

**Q: What if I don't want interactive prompts?**

A: Provide both the directory and `--ffmpeg` path as arguments to skip all prompts.

## Contributing

Feel free to add implementations in other languages or improve existing ones!

## License

MIT License - Feel free to use and modify as needed.

## Credits

Original PowerShell implementation for NVR video combination workflows.
Cross-platform versions created for broader usability.
