package main

import (
	"bufio"
	"flag"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"runtime"
	"sort"
	"strings"
)

const (
	colorReset  = "\033[0m"
	colorGreen  = "\033[32m"
	colorYellow = "\033[33m"
	colorRed    = "\033[31m"
)

// Config holds the application configuration
type Config struct {
	VideoDir   string
	OutputFile string
	FFmpegPath string
	KeepConcat bool
}

func main() {
	config := parseFlags()

	if err := run(config); err != nil {
		fmt.Fprintf(os.Stderr, "%s✗ Error: %v%s\n", colorRed, err, colorReset)
		os.Exit(1)
	}
}

func parseFlags() Config {
	var config Config

	defaultVideoDir := getDefaultVideoDir()

	flag.StringVar(&config.VideoDir, "dir", "", "Directory containing MP4 files")
	flag.StringVar(&config.OutputFile, "output", "Combined_Output.mp4", "Output filename")
	flag.StringVar(&config.FFmpegPath, "ffmpeg", "", "Path to ffmpeg executable (auto-detected if not specified)")
	flag.BoolVar(&config.KeepConcat, "keep-concat", false, "Keep the concat.txt file after completion")

	var showHelp bool
	flag.BoolVar(&showHelp, "h", false, "Show help message")
	flag.BoolVar(&showHelp, "help", false, "Show help message")

	flag.Usage = func() {
		fmt.Fprintf(os.Stderr, "NVR Video Combiner - Go Version\n\n")
		fmt.Fprintf(os.Stderr, "Usage: %s [OPTIONS] [VIDEO_DIR]\n\n", os.Args[0])
		fmt.Fprintf(os.Stderr, "Combines multiple MP4 video files using ffmpeg (no re-encoding)\n\n")
		fmt.Fprintf(os.Stderr, "Arguments:\n")
		fmt.Fprintf(os.Stderr, "    VIDEO_DIR           Directory containing MP4 files (default: %s)\n\n", defaultVideoDir)
		fmt.Fprintf(os.Stderr, "Options:\n")
		fmt.Fprintf(os.Stderr, "    -output FILE        Output filename (default: Combined_Output.mp4)\n")
		fmt.Fprintf(os.Stderr, "    -ffmpeg PATH        Path to ffmpeg executable (auto-detected if not specified)\n")
		fmt.Fprintf(os.Stderr, "    -keep-concat        Keep the concat.txt file after completion\n")
		fmt.Fprintf(os.Stderr, "    -h, -help           Show this help message\n")
		fmt.Fprintf(os.Stderr, "\nExamples:\n")
		fmt.Fprintf(os.Stderr, "  %s /path/to/videos\n", os.Args[0])
		fmt.Fprintf(os.Stderr, "  %s -output MyVideo.mp4 /path/to/videos\n", os.Args[0])
		fmt.Fprintf(os.Stderr, "  %s -ffmpeg /usr/local/bin/ffmpeg /path/to/videos\n", os.Args[0])
	}

	flag.Parse()

	if showHelp {
		flag.Usage()
		os.Exit(0)
	}

	// Handle positional argument
	if flag.NArg() > 0 {
		config.VideoDir = flag.Arg(0)
	}

	return config
}

func getDefaultVideoDir() string {
	if runtime.GOOS == "windows" {
		return filepath.Join(os.Getenv("USERPROFILE"), "Videos", "NVR")
	}
	home, _ := os.UserHomeDir()
	return filepath.Join(home, "Videos", "NVR")
}

func promptInput(prompt string) string {
	fmt.Print(prompt)
	reader := bufio.NewReader(os.Stdin)
	input, _ := reader.ReadString('\n')
	return strings.TrimSpace(input)
}

func run(config Config) error {
	defaultDir := getDefaultVideoDir()

	// Prompt for video directory if not specified
	if config.VideoDir == "" {
		input := promptInput(fmt.Sprintf("Enter video directory path [%s]: ", defaultDir))
		if input == "" {
			config.VideoDir = defaultDir
		} else {
			config.VideoDir = input
		}
	}

	// Validate video directory
	if _, err := os.Stat(config.VideoDir); os.IsNotExist(err) {
		return fmt.Errorf("directory does not exist: %s", config.VideoDir)
	}

	// Find ffmpeg with comprehensive search and user prompts
	ffmpegPath := config.FFmpegPath
	if ffmpegPath == "" {
		var err error
		ffmpegPath, err = findFFmpeg()
		if err != nil {
			ffmpegPath, err = promptForFFmpeg()
			if err != nil {
				return err
			}
		} else {
			fmt.Printf("%sFound ffmpeg: %s%s\n", colorGreen, ffmpegPath, colorReset)
		}
	}

	// Verify ffmpeg is executable
	if _, err := os.Stat(ffmpegPath); os.IsNotExist(err) {
		// Check if it's in PATH
		if _, err := exec.LookPath(ffmpegPath); err != nil {
			return fmt.Errorf("ffmpeg not executable: %s", ffmpegPath)
		}
	}

	// Setup paths
	concatFile := filepath.Join(config.VideoDir, "concat.txt")
	outputPath := filepath.Join(config.VideoDir, config.OutputFile)

	// Find video files
	videoFiles, err := findVideoFiles(config.VideoDir)
	if err != nil {
		return err
	}

	if len(videoFiles) == 0 {
		return fmt.Errorf("no .mp4 files found in %s", config.VideoDir)
	}

	// Display found videos
	fmt.Printf("%sFound %d video file(s):%s\n", colorGreen, len(videoFiles), colorReset)
	for _, video := range videoFiles {
		fmt.Printf("  - %s\n", filepath.Base(video))
	}

	// Create concat file
	fmt.Printf("\n%sCreating concat file: %s%s\n", colorYellow, concatFile, colorReset)
	if err := createConcatFile(videoFiles, concatFile); err != nil {
		return fmt.Errorf("failed to create concat file: %v", err)
	}

	// Run ffmpeg
	fmt.Printf("\n%sRunning ffmpeg...%s\n", colorYellow, colorReset)
	fmt.Printf("Output will be: %s%s%s\n\n", colorGreen, outputPath, colorReset)

	success := true
	if err := runFFmpeg(ffmpegPath, concatFile, outputPath); err != nil {
		fmt.Printf("\n%s✗ Error running ffmpeg%s\n", colorRed, colorReset)
		success = false
	} else {
		fmt.Printf("\n%s✓ Successfully created: %s%s\n", colorGreen, outputPath, colorReset)
	}

	// Cleanup concat file
	if !config.KeepConcat {
		os.Remove(concatFile)
		fmt.Printf("Cleaned up: %s\n", concatFile)
	}

	if !success {
		return fmt.Errorf("ffmpeg failed")
	}
	return nil
}

func findFFmpeg() (string, error) {
	fmt.Fprintf(os.Stderr, "%sSearching for ffmpeg...%s\n", colorYellow, colorReset)

	// Try to find in PATH
	path, err := exec.LookPath("ffmpeg")
	if err == nil {
		return path, nil
	}

	// Deep search in common locations
	var searchPaths []string
	var ffmpegName string

	if runtime.GOOS == "windows" {
		ffmpegName = "ffmpeg.exe"
		searchPaths = []string{
			"C:\\ffmpeg",
			"C:\\Program Files\\ffmpeg",
			"C:\\Program Files (x86)\\ffmpeg",
			filepath.Join(os.Getenv("LOCALAPPDATA"), "Microsoft", "WinGet", "Packages"),
			filepath.Join(os.Getenv("USERPROFILE"), "scoop", "apps", "ffmpeg"),
			filepath.Join(os.Getenv("ChocolateyInstall"), "bin"),
			"C:\\tools",
		}
	} else {
		ffmpegName = "ffmpeg"
		home, _ := os.UserHomeDir()
		searchPaths = []string{
			"/usr/bin",
			"/usr/local/bin",
			"/opt",
			"/snap/bin",
			filepath.Join(home, ".local", "bin"),
			filepath.Join(home, "bin"),
			"/opt/homebrew/bin",
			"/usr/share",
			"/Applications",
		}
	}

	fmt.Fprintf(os.Stderr, "%sPerforming deep search in common directories...%s\n", colorYellow, colorReset)
	for _, base := range searchPaths {
		if _, err := os.Stat(base); err == nil {
			err := filepath.Walk(base, func(path string, info os.FileInfo, err error) error {
				if err != nil {
					return nil // Skip errors
				}
				if !info.IsDir() && info.Name() == ffmpegName {
					return fmt.Errorf("found:%s", path) // Use error to break out with path
				}
				return nil
			})
			if err != nil && strings.HasPrefix(err.Error(), "found:") {
				return strings.TrimPrefix(err.Error(), "found:"), nil
			}
		}
	}

	return "", fmt.Errorf("ffmpeg not found in PATH or common locations")
}

func promptForFFmpeg() (string, error) {
	fmt.Printf("%sffmpeg not found after comprehensive search%s\n", colorRed, colorReset)
	userPath := promptInput("Enter path to ffmpeg executable (or press Enter to install): ")

	if userPath != "" {
		if info, err := os.Stat(userPath); err == nil && !info.IsDir() {
			fmt.Printf("%sUsing: %s%s\n", colorGreen, userPath, colorReset)
			return userPath, nil
		}
		return "", fmt.Errorf("not a valid executable: %s", userPath)
	}

	reply := promptInput("Install ffmpeg now? (y/n): ")
	if strings.ToLower(reply) == "y" || strings.ToLower(reply) == "yes" {
		return installFFmpeg()
	}

	return "", fmt.Errorf("ffmpeg required to continue")
}

func installFFmpeg() (string, error) {
	switch runtime.GOOS {
	case "windows":
		if _, err := exec.LookPath("winget"); err == nil {
			fmt.Printf("%sInstalling ffmpeg via winget...%s\n", colorYellow, colorReset)
			cmd := exec.Command("winget", "install", "ffmpeg")
			cmd.Stdout = os.Stdout
			cmd.Stderr = os.Stderr
			cmd.Run()
		} else {
			fmt.Println("Please download ffmpeg from https://ffmpeg.org/download.html")
			return "", fmt.Errorf("cannot auto-install")
		}
	case "darwin":
		if _, err := exec.LookPath("brew"); err == nil {
			fmt.Printf("%sInstalling ffmpeg via Homebrew...%s\n", colorYellow, colorReset)
			cmd := exec.Command("brew", "install", "ffmpeg")
			cmd.Stdout = os.Stdout
			cmd.Stderr = os.Stderr
			if err := cmd.Run(); err != nil {
				return "", fmt.Errorf("homebrew install failed: %v", err)
			}
		} else {
			return "", fmt.Errorf("please install Homebrew first or download ffmpeg manually")
		}
	default:
		// Linux
		if _, err := exec.LookPath("apt-get"); err == nil {
			exec.Command("sudo", "apt-get", "update").Run()
			cmd := exec.Command("sudo", "apt-get", "install", "-y", "ffmpeg")
			cmd.Stdout = os.Stdout
			cmd.Stderr = os.Stderr
			cmd.Run()
		} else if _, err := exec.LookPath("yum"); err == nil {
			cmd := exec.Command("sudo", "yum", "install", "-y", "ffmpeg")
			cmd.Stdout = os.Stdout
			cmd.Stderr = os.Stderr
			cmd.Run()
		} else {
			return "", fmt.Errorf("cannot auto-install. Please install manually")
		}
	}

	// Try to find it again
	path, err := findFFmpeg()
	if err != nil {
		return "", fmt.Errorf("installation failed")
	}
	return path, nil
}

func findVideoFiles(dir string) ([]string, error) {
	pattern := filepath.Join(dir, "*.mp4")
	files, err := filepath.Glob(pattern)
	if err != nil {
		return nil, err
	}

	sort.Strings(files)
	return files, nil
}

func createConcatFile(videoFiles []string, concatFile string) error {
	f, err := os.Create(concatFile)
	if err != nil {
		return err
	}
	defer f.Close()

	for _, video := range videoFiles {
		absPath, err := filepath.Abs(video)
		if err != nil {
			return err
		}

		// Escape single quotes
		escapedPath := strings.ReplaceAll(absPath, "'", "'\\''")
		_, err = fmt.Fprintf(f, "file '%s'\n", escapedPath)
		if err != nil {
			return err
		}
	}

	return nil
}

func runFFmpeg(ffmpegPath, concatFile, outputPath string) error {
	cmd := exec.Command(
		ffmpegPath,
		"-f", "concat",
		"-safe", "0",
		"-i", concatFile,
		"-c", "copy",
		outputPath,
	)

	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	return cmd.Run()
}
