#!/usr/bin/env node

/**
 * NVR Video Combiner - Node.js Version
 * Combines multiple MP4 video files using ffmpeg's concat demuxer (no re-encoding)
 */

const fs = require('fs').promises;
const fsSync = require('fs');
const path = require('path');
const { spawn, execSync } = require('child_process');
const { platform, homedir } = require('os');
const readline = require('readline');

// ANSI color codes
const colors = {
    reset: '\x1b[0m',
    green: '\x1b[32m',
    yellow: '\x1b[33m',
    red: '\x1b[31m',
};

// Default configuration
function getDefaultVideoDir() {
    if (platform() === 'win32') {
        return path.join(process.env.USERPROFILE || '', 'Videos', 'NVR');
    }
    return path.join(homedir(), 'Videos', 'NVR');
}

// Prompt user for input
function prompt(question) {
    const rl = readline.createInterface({
        input: process.stdin,
        output: process.stdout
    });
    return new Promise((resolve) => {
        rl.question(question, (answer) => {
            rl.close();
            resolve(answer.trim());
        });
    });
}

// Parse command line arguments
function parseArgs() {
    const args = process.argv.slice(2);
    const config = {
        videoDir: null,
        outputFile: 'Combined_Output.mp4',
        ffmpegPath: null,
        keepConcat: false,
    };

    for (let i = 0; i < args.length; i++) {
        switch (args[i]) {
            case '-h':
            case '--help':
                printUsage();
                process.exit(0);
                break;
            case '-d':
            case '--dir':
                if (i + 1 >= args.length) {
                    console.error('Error: --dir requires a value');
                    process.exit(1);
                }
                config.videoDir = args[++i];
                break;
            case '-o':
            case '--output':
                if (i + 1 >= args.length) {
                    console.error('Error: --output requires a value');
                    process.exit(1);
                }
                config.outputFile = args[++i];
                break;
            case '--ffmpeg':
                if (i + 1 >= args.length) {
                    console.error('Error: --ffmpeg requires a value');
                    process.exit(1);
                }
                config.ffmpegPath = args[++i];
                break;
            case '--keep-concat':
                config.keepConcat = true;
                break;
            default:
                if (args[i].startsWith('-')) {
                    console.error(`Error: Unknown option: ${args[i]}`);
                    printUsage();
                    process.exit(1);
                }
                // Positional argument - video directory
                config.videoDir = args[i];
        }
    }

    return config;
}

function printUsage() {
    const defaultDir = getDefaultVideoDir();
    console.log('NVR Video Combiner - Node.js Version\n');
    console.log('Usage: node join_videos.js [OPTIONS] [VIDEO_DIR]\n');
    console.log('Combines multiple MP4 video files using ffmpeg (no re-encoding)\n');
    console.log('Arguments:');
    console.log(`    VIDEO_DIR           Directory containing MP4 files (default: ${defaultDir})\n`);
    console.log('Options:');
    console.log('  -o, --output FILE   Output filename (default: Combined_Output.mp4)');
    console.log('  --ffmpeg PATH       Path to ffmpeg executable (auto-detected if not specified)');
    console.log('  --keep-concat       Keep the concat.txt file after completion');
    console.log('  -h, --help          Show this help message');
    console.log('\nExamples:');
    console.log('  node join_videos.js /path/to/videos');
    console.log('  node join_videos.js -o MyVideo.mp4 /path/to/videos');
    console.log('  node join_videos.js --ffmpeg /usr/local/bin/ffmpeg /path/to/videos');
}

// Find ffmpeg with comprehensive search
async function findFFmpeg() {
    console.error(`${colors.yellow}Searching for ffmpeg...${colors.reset}`);

    // Check PATH first
    try {
        const command = platform() === 'win32' ? 'where' : 'which';
        const result = execSync(`${command} ffmpeg`, { encoding: 'utf8', stdio: ['pipe', 'pipe', 'pipe'] });
        return result.trim().split('\n')[0];
    } catch (e) {
        // Not in PATH, continue to deep search
    }

    // Deep search in common locations
    let searchPaths;
    let ffmpegName;

    if (platform() === 'win32') {
        ffmpegName = 'ffmpeg.exe';
        searchPaths = [
            'C:\\ffmpeg',
            'C:\\Program Files\\ffmpeg',
            'C:\\Program Files (x86)\\ffmpeg',
            path.join(process.env.LOCALAPPDATA || '', 'Microsoft', 'WinGet', 'Packages'),
            path.join(process.env.USERPROFILE || '', 'scoop', 'apps', 'ffmpeg'),
            path.join(process.env.ChocolateyInstall || 'C:\\ProgramData\\chocolatey', 'bin'),
            'C:\\tools',
        ];
    } else {
        ffmpegName = 'ffmpeg';
        searchPaths = [
            '/usr/bin',
            '/usr/local/bin',
            '/opt',
            '/snap/bin',
            path.join(homedir(), '.local', 'bin'),
            path.join(homedir(), 'bin'),
            '/opt/homebrew/bin',
            '/usr/share',
            '/Applications',
        ];
    }

    console.error(`${colors.yellow}Performing deep search in common directories...${colors.reset}`);

    for (const base of searchPaths) {
        try {
            const found = await findFileRecursive(base, ffmpegName);
            if (found) {
                return found;
            }
        } catch (e) {
            continue;
        }
    }

    return null;
}

// Recursive file search
async function findFileRecursive(dir, name) {
    try {
        const entries = await fs.readdir(dir, { withFileTypes: true });
        for (const entry of entries) {
            const fullPath = path.join(dir, entry.name);
            if (entry.isFile() && entry.name === name) {
                return fullPath;
            }
            if (entry.isDirectory()) {
                try {
                    const found = await findFileRecursive(fullPath, name);
                    if (found) return found;
                } catch (e) {
                    continue;
                }
            }
        }
    } catch (e) {
        return null;
    }
    return null;
}

// Prompt user for ffmpeg path or offer installation
async function promptForFFmpeg() {
    console.log(`${colors.red}ffmpeg not found after comprehensive search${colors.reset}`);
    const userPath = await prompt('Enter path to ffmpeg executable (or press Enter to install): ');

    if (userPath) {
        try {
            await fs.access(userPath, fsSync.constants.X_OK);
            console.log(`${colors.green}Using: ${userPath}${colors.reset}`);
            return userPath;
        } catch (e) {
            throw new Error(`Not a valid executable: ${userPath}`);
        }
    }

    const reply = await prompt('Install ffmpeg now? (y/n): ');
    if (reply.toLowerCase() === 'y' || reply.toLowerCase() === 'yes') {
        return await installFFmpeg();
    }

    throw new Error('ffmpeg required to continue');
}

// Attempt to install ffmpeg
async function installFFmpeg() {
    if (platform() === 'win32') {
        try {
            execSync('where winget', { stdio: 'pipe' });
            console.log(`${colors.yellow}Installing ffmpeg via winget...${colors.reset}`);
            execSync('winget install ffmpeg', { stdio: 'inherit' });
        } catch (e) {
            console.log('Please download ffmpeg from https://ffmpeg.org/download.html');
            throw new Error('Cannot auto-install');
        }
    } else if (platform() === 'darwin') {
        try {
            execSync('which brew', { stdio: 'pipe' });
            console.log(`${colors.yellow}Installing ffmpeg via Homebrew...${colors.reset}`);
            execSync('brew install ffmpeg', { stdio: 'inherit' });
        } catch (e) {
            throw new Error('Please install Homebrew first or download ffmpeg manually');
        }
    } else {
        // Linux
        try {
            execSync('which apt-get', { stdio: 'pipe' });
            console.log(`${colors.yellow}Installing ffmpeg via apt...${colors.reset}`);
            execSync('sudo apt-get update && sudo apt-get install -y ffmpeg', { stdio: 'inherit' });
        } catch (e) {
            try {
                execSync('which yum', { stdio: 'pipe' });
                execSync('sudo yum install -y ffmpeg', { stdio: 'inherit' });
            } catch (e2) {
                throw new Error('Cannot auto-install. Please install manually');
            }
        }
    }

    // Try to find it again
    const ffmpeg = await findFFmpeg();
    if (!ffmpeg) {
        throw new Error('Installation failed');
    }
    return ffmpeg;
}

// Find all MP4 files in directory
async function findVideoFiles(dir) {
    const entries = await fs.readdir(dir, { withFileTypes: true });
    const videos = entries
        .filter(entry => entry.isFile() && entry.name.toLowerCase().endsWith('.mp4'))
        .map(entry => path.join(dir, entry.name))
        .sort();

    return videos;
}

// Create concat file
async function createConcatFile(videoFiles, concatFile) {
    const lines = videoFiles.map(video => {
        const absolutePath = path.resolve(video);
        // Escape single quotes
        const escapedPath = absolutePath.replace(/'/g, "'\\''");
        return `file '${escapedPath}'`;
    });

    await fs.writeFile(concatFile, lines.join('\n') + '\n', 'utf8');
}

// Run ffmpeg
function runFFmpeg(ffmpegPath, concatFile, outputPath) {
    return new Promise((resolve, reject) => {
        const args = [
            '-f', 'concat',
            '-safe', '0',
            '-i', concatFile,
            '-c', 'copy',
            outputPath
        ];

        const ffmpeg = spawn(ffmpegPath, args, {
            stdio: 'inherit'
        });

        ffmpeg.on('close', (code) => {
            if (code === 0) {
                resolve();
            } else {
                reject(new Error(`ffmpeg exited with code ${code}`));
            }
        });

        ffmpeg.on('error', (err) => {
            reject(err);
        });
    });
}

// Main function
async function main() {
    try {
        const config = parseArgs();
        const defaultDir = getDefaultVideoDir();

        // Prompt for video directory if not specified
        let videoDir = config.videoDir;
        if (!videoDir) {
            const input = await prompt(`Enter video directory path [${defaultDir}]: `);
            videoDir = input || defaultDir;
        }

        // Validate video directory
        try {
            const stats = await fs.stat(videoDir);
            if (!stats.isDirectory()) {
                throw new Error(`Not a directory: ${videoDir}`);
            }
        } catch (e) {
            throw new Error(`Directory does not exist: ${videoDir}`);
        }

        // Find ffmpeg with comprehensive search and user prompts
        let ffmpegPath = config.ffmpegPath;
        if (!ffmpegPath) {
            ffmpegPath = await findFFmpeg();
            if (ffmpegPath) {
                console.log(`${colors.green}Found ffmpeg: ${ffmpegPath}${colors.reset}`);
            } else {
                ffmpegPath = await promptForFFmpeg();
            }
        }

        // Setup paths
        const concatFile = path.join(videoDir, 'concat.txt');
        const outputPath = path.join(videoDir, config.outputFile);

        // Find video files
        const videoFiles = await findVideoFiles(videoDir);
        if (videoFiles.length === 0) {
            throw new Error(`No .mp4 files found in ${videoDir}`);
        }

        // Display found videos
        console.log(`${colors.green}Found ${videoFiles.length} video file(s):${colors.reset}`);
        videoFiles.forEach(video => {
            console.log(`  - ${path.basename(video)}`);
        });

        // Create concat file
        console.log(`\n${colors.yellow}Creating concat file: ${concatFile}${colors.reset}`);
        await createConcatFile(videoFiles, concatFile);

        // Run ffmpeg
        console.log(`\n${colors.yellow}Running ffmpeg...${colors.reset}`);
        console.log(`Output will be: ${colors.green}${outputPath}${colors.reset}\n`);

        let success = true;
        try {
            await runFFmpeg(ffmpegPath, concatFile, outputPath);
            console.log(`\n${colors.green}✓ Successfully created: ${outputPath}${colors.reset}`);
        } catch (e) {
            console.error(`\n${colors.red}✗ Error running ffmpeg${colors.reset}`);
            success = false;
        }

        // Cleanup concat file
        if (!config.keepConcat) {
            await fs.unlink(concatFile);
            console.log(`Cleaned up: ${concatFile}`);
        }

        process.exit(success ? 0 : 1);
    } catch (error) {
        console.error(`${colors.red}✗ Error: ${error.message}${colors.reset}`);
        process.exit(1);
    }
}

// Run if called directly
if (require.main === module) {
    main();
}

module.exports = { findFFmpeg, findVideoFiles, createConcatFile, runFFmpeg };
