#!/usr/bin/env python3
"""
NVR Video Combiner - Python Version
Combines multiple MP4 video files using ffmpeg's concat demuxer (no re-encoding)
"""

import os
import sys
import subprocess
import argparse
from pathlib import Path
from shutil import which

# ANSI colors
RED = '\033[0;31m'
GREEN = '\033[0;32m'
YELLOW = '\033[1;33m'
NC = '\033[0m'  # No Color


def get_default_video_dir():
    """Get platform-specific default video directory"""
    if sys.platform == 'win32':
        return Path(os.environ.get('USERPROFILE', '')) / 'Videos' / 'NVR'
    return Path.home() / 'Videos' / 'NVR'


def find_ffmpeg():
    """Find ffmpeg with comprehensive search"""
    print(f"{YELLOW}Searching for ffmpeg...{NC}", file=sys.stderr)

    # Check PATH first
    if which("ffmpeg"):
        return "ffmpeg"

    # Deep search in common locations
    if sys.platform == 'win32':
        search_paths = [
            Path("C:/ffmpeg"),
            Path("C:/Program Files/ffmpeg"),
            Path("C:/Program Files (x86)/ffmpeg"),
            Path(os.environ.get('LOCALAPPDATA', '')) / 'Microsoft' / 'WinGet' / 'Packages',
            Path(os.environ.get('USERPROFILE', '')) / 'scoop' / 'apps' / 'ffmpeg',
            Path(os.environ.get('ChocolateyInstall', 'C:/ProgramData/chocolatey')) / 'bin',
            Path("C:/tools"),
        ]
        ffmpeg_name = "ffmpeg.exe"
    else:
        search_paths = [
            Path("/usr/bin"),
            Path("/usr/local/bin"),
            Path("/opt"),
            Path("/snap/bin"),
            Path.home() / ".local" / "bin",
            Path.home() / "bin",
            Path("/opt/homebrew/bin"),
            Path("/usr/share"),
            Path("/Applications"),
        ]
        ffmpeg_name = "ffmpeg"

    print(f"{YELLOW}Performing deep search in common directories...{NC}", file=sys.stderr)
    for base in search_paths:
        if base.exists():
            for found in base.rglob(ffmpeg_name):
                if found.is_file() and os.access(found, os.X_OK):
                    return str(found)

    return None


def prompt_for_ffmpeg():
    """Prompt user for ffmpeg path or offer installation"""
    print(f"{RED}ffmpeg not found after comprehensive search{NC}")
    user_path = input("Enter path to ffmpeg executable (or press Enter to install): ").strip()

    if user_path:
        if os.path.isfile(user_path) and os.access(user_path, os.X_OK):
            print(f"{GREEN}Using: {user_path}{NC}")
            return user_path
        else:
            print(f"{RED}Error: Not a valid executable: {user_path}{NC}", file=sys.stderr)
            sys.exit(1)
    else:
        reply = input("Install ffmpeg now? (y/n): ").strip().lower()
        if reply in ('y', 'yes'):
            return install_ffmpeg()
        else:
            print(f"{RED}ffmpeg required to continue{NC}", file=sys.stderr)
            sys.exit(1)


def install_ffmpeg():
    """Attempt to install ffmpeg based on platform"""
    if sys.platform == 'win32':
        # Try winget
        if which("winget"):
            print(f"{YELLOW}Installing ffmpeg via winget...{NC}")
            subprocess.run(["winget", "install", "ffmpeg"], check=False)
        else:
            print("Please download ffmpeg from https://ffmpeg.org/download.html")
            sys.exit(1)
    elif sys.platform == 'darwin':
        if which("brew"):
            print(f"{YELLOW}Installing ffmpeg via Homebrew...{NC}")
            subprocess.run(["brew", "install", "ffmpeg"], check=True)
        else:
            print("Please install Homebrew first or download ffmpeg manually")
            sys.exit(1)
    else:
        # Linux
        if which("apt-get"):
            subprocess.run(["sudo", "apt-get", "update"], check=True)
            subprocess.run(["sudo", "apt-get", "install", "-y", "ffmpeg"], check=True)
        elif which("yum"):
            subprocess.run(["sudo", "yum", "install", "-y", "ffmpeg"], check=True)
        elif which("pacman"):
            subprocess.run(["sudo", "pacman", "-S", "--noconfirm", "ffmpeg"], check=True)
        else:
            print(f"{RED}Error: Cannot auto-install. Please install manually.{NC}", file=sys.stderr)
            sys.exit(1)

    # Try to find it again
    ffmpeg = find_ffmpeg()
    if not ffmpeg:
        print(f"{RED}Error: Installation failed{NC}", file=sys.stderr)
        sys.exit(1)
    return ffmpeg


def create_concat_file(video_dir, concat_file):
    """Create concat.txt file with list of videos to combine"""
    # Get all .mp4 files sorted by name
    video_files = sorted(Path(video_dir).glob("*.mp4"))

    if not video_files:
        print(f"{RED}Error: No .mp4 files found in {video_dir}{NC}")
        sys.exit(1)

    print(f"{GREEN}Found {len(video_files)} video file(s):{NC}")
    for video in video_files:
        print(f"  - {video.name}")

    # Write concat file
    with open(concat_file, 'w', encoding='utf-8') as f:
        for video in video_files:
            # Escape single quotes in path
            escaped_path = str(video.absolute()).replace("'", "'\\''")
            f.write(f"file '{escaped_path}'\n")

    print(f"\n{YELLOW}Creating concat file: {concat_file}{NC}")
    return len(video_files)


def combine_videos(ffmpeg_path, concat_file, output_file):
    """Run ffmpeg to combine videos"""
    cmd = [
        ffmpeg_path,
        "-f", "concat",
        "-safe", "0",
        "-i", str(concat_file),
        "-c", "copy",
        str(output_file)
    ]

    print(f"\n{YELLOW}Running ffmpeg...{NC}")
    print(f"Output will be: {GREEN}{output_file}{NC}\n")

    try:
        subprocess.run(cmd, check=True)
        print(f"\n{GREEN}✓ Successfully created: {output_file}{NC}")
        return True
    except subprocess.CalledProcessError:
        print(f"\n{RED}✗ Error running ffmpeg{NC}")
        return False
    except FileNotFoundError:
        print(f"\n{RED}✗ Error: ffmpeg not found at {ffmpeg_path}{NC}")
        print("Please install ffmpeg or specify its path with --ffmpeg")
        return False


def main():
    default_dir = get_default_video_dir()

    parser = argparse.ArgumentParser(
        description="Combine multiple MP4 video files using ffmpeg (no re-encoding)"
    )
    parser.add_argument(
        "video_dir",
        nargs="?",
        default=None,
        help=f"Directory containing MP4 files to combine (default: {default_dir})"
    )
    parser.add_argument(
        "-o", "--output",
        default="Combined_Output.mp4",
        help="Output filename (default: Combined_Output.mp4)"
    )
    parser.add_argument(
        "--ffmpeg",
        help="Path to ffmpeg executable (auto-detected if not specified)"
    )
    parser.add_argument(
        "--keep-concat",
        action="store_true",
        help="Keep the concat.txt file after completion"
    )

    args = parser.parse_args()

    # Prompt for video directory if not specified
    video_dir = args.video_dir
    if not video_dir:
        user_input = input(f"Enter video directory path [{default_dir}]: ").strip()
        video_dir = user_input if user_input else str(default_dir)

    video_dir = Path(video_dir)

    # Validate video directory
    if not video_dir.exists():
        print(f"{RED}Error: Directory does not exist: {video_dir}{NC}")
        sys.exit(1)

    # Find ffmpeg with comprehensive search and user prompts
    if args.ffmpeg:
        ffmpeg_path = args.ffmpeg
    else:
        ffmpeg_path = find_ffmpeg()
        if ffmpeg_path:
            print(f"{GREEN}Found ffmpeg: {ffmpeg_path}{NC}")
        else:
            ffmpeg_path = prompt_for_ffmpeg()

    if not os.path.isfile(ffmpeg_path) and not which(ffmpeg_path):
        print(f"{RED}Error: ffmpeg not executable: {ffmpeg_path}{NC}", file=sys.stderr)
        sys.exit(1)

    # Setup paths
    concat_file = video_dir / "concat.txt"
    output_file = video_dir / args.output

    # Create concat file and combine
    create_concat_file(video_dir, concat_file)
    success = combine_videos(ffmpeg_path, concat_file, output_file)

    # Cleanup concat file unless requested to keep
    if not args.keep_concat and concat_file.exists():
        concat_file.unlink()
        print(f"Cleaned up: {concat_file}")

    sys.exit(0 if success else 1)


if __name__ == "__main__":
    main()
