use std::env;
use std::fs::{self, File};
use std::io::{self, Write, BufRead};
use std::path::{Path, PathBuf};
use std::process::{Command, exit};

const COLOR_RESET: &str = "\x1b[0m";
const COLOR_GREEN: &str = "\x1b[32m";
const COLOR_YELLOW: &str = "\x1b[33m";
const COLOR_RED: &str = "\x1b[31m";

struct Config {
    video_dir: Option<PathBuf>,
    output_file: String,
    ffmpeg_path: Option<String>,
    keep_concat: bool,
}

fn main() {
    let config = parse_args();

    if let Err(e) = run(config) {
        eprintln!("{}✗ Error: {}{}", COLOR_RED, e, COLOR_RESET);
        exit(1);
    }
}

fn parse_args() -> Config {
    let args: Vec<String> = env::args().collect();
    let mut config = Config {
        video_dir: None,
        output_file: "Combined_Output.mp4".to_string(),
        ffmpeg_path: None,
        keep_concat: false,
    };

    let mut i = 1;
    while i < args.len() {
        match args[i].as_str() {
            "-d" | "--dir" => {
                if i + 1 < args.len() {
                    config.video_dir = Some(PathBuf::from(&args[i + 1]));
                    i += 2;
                } else {
                    eprintln!("Error: --dir requires a value");
                    print_usage(&args[0]);
                    exit(1);
                }
            }
            "-o" | "--output" => {
                if i + 1 < args.len() {
                    config.output_file = args[i + 1].clone();
                    i += 2;
                } else {
                    eprintln!("Error: --output requires a value");
                    print_usage(&args[0]);
                    exit(1);
                }
            }
            "--ffmpeg" => {
                if i + 1 < args.len() {
                    config.ffmpeg_path = Some(args[i + 1].clone());
                    i += 2;
                } else {
                    eprintln!("Error: --ffmpeg requires a value");
                    print_usage(&args[0]);
                    exit(1);
                }
            }
            "--keep-concat" => {
                config.keep_concat = true;
                i += 1;
            }
            "-h" | "--help" => {
                print_usage(&args[0]);
                exit(0);
            }
            arg if !arg.starts_with('-') => {
                // Positional argument - video directory
                config.video_dir = Some(PathBuf::from(arg));
                i += 1;
            }
            _ => {
                eprintln!("Error: Unknown option: {}", args[i]);
                print_usage(&args[0]);
                exit(1);
            }
        }
    }

    config
}

fn print_usage(program_name: &str) {
    let default_dir = get_default_video_dir();
    println!("NVR Video Combiner - Rust Version\n");
    println!("Usage: {} [OPTIONS] [VIDEO_DIR]\n", program_name);
    println!("Combines multiple MP4 video files using ffmpeg (no re-encoding)\n");
    println!("Arguments:");
    println!("    VIDEO_DIR           Directory containing MP4 files (default: {:?})\n", default_dir);
    println!("Options:");
    println!("  -o, --output FILE   Output filename (default: Combined_Output.mp4)");
    println!("  --ffmpeg PATH       Path to ffmpeg executable (auto-detected if not specified)");
    println!("  --keep-concat       Keep the concat.txt file after completion");
    println!("  -h, --help          Show this help message");
    println!("\nExamples:");
    println!("  {} /path/to/videos", program_name);
    println!("  {} -o MyVideo.mp4 /path/to/videos", program_name);
    println!("  {} --ffmpeg /usr/local/bin/ffmpeg /path/to/videos", program_name);
}

fn get_default_video_dir() -> PathBuf {
    if cfg!(windows) {
        PathBuf::from(env::var("USERPROFILE").unwrap_or_default())
            .join("Videos")
            .join("NVR")
    } else {
        let home = env::var("HOME").unwrap_or_else(|_| ".".to_string());
        PathBuf::from(home).join("Videos").join("NVR")
    }
}

fn prompt_input(prompt: &str) -> String {
    print!("{}", prompt);
    io::stdout().flush().unwrap();
    let mut input = String::new();
    io::stdin().lock().read_line(&mut input).unwrap();
    input.trim().to_string()
}

fn run(mut config: Config) -> Result<(), Box<dyn std::error::Error>> {
    let default_dir = get_default_video_dir();

    // Prompt for video directory if not specified
    if config.video_dir.is_none() {
        let input = prompt_input(&format!("Enter video directory path [{:?}]: ", default_dir));
        if input.is_empty() {
            config.video_dir = Some(default_dir);
        } else {
            config.video_dir = Some(PathBuf::from(input));
        }
    }

    let video_dir = config.video_dir.unwrap();

    // Validate video directory
    if !video_dir.exists() {
        return Err(format!("Directory does not exist: {:?}", video_dir).into());
    }

    // Find ffmpeg with comprehensive search and user prompts
    let ffmpeg_path = match config.ffmpeg_path {
        Some(path) => path,
        None => {
            match find_ffmpeg() {
                Ok(path) => {
                    println!("{}Found ffmpeg: {}{}", COLOR_GREEN, path, COLOR_RESET);
                    path
                }
                Err(_) => prompt_for_ffmpeg()?,
            }
        }
    };

    // Setup paths
    let concat_file = video_dir.join("concat.txt");
    let output_path = video_dir.join(&config.output_file);

    // Find video files
    let mut video_files = find_video_files(&video_dir)?;
    if video_files.is_empty() {
        return Err(format!("No .mp4 files found in {:?}", video_dir).into());
    }

    video_files.sort();

    // Display found videos
    println!("{}Found {} video file(s):{}", COLOR_GREEN, video_files.len(), COLOR_RESET);
    for video in &video_files {
        if let Some(name) = video.file_name() {
            println!("  - {}", name.to_string_lossy());
        }
    }

    // Create concat file
    println!("\n{}Creating concat file: {:?}{}", COLOR_YELLOW, concat_file, COLOR_RESET);
    create_concat_file(&video_files, &concat_file)?;

    // Run ffmpeg
    println!("\n{}Running ffmpeg...{}", COLOR_YELLOW, COLOR_RESET);
    println!("Output will be: {}{:?}{}\n", COLOR_GREEN, output_path, COLOR_RESET);

    let success = match run_ffmpeg(&ffmpeg_path, &concat_file, &output_path) {
        Ok(()) => {
            println!("\n{}✓ Successfully created: {:?}{}", COLOR_GREEN, output_path, COLOR_RESET);
            true
        }
        Err(_) => {
            eprintln!("\n{}✗ Error running ffmpeg{}", COLOR_RED, COLOR_RESET);
            false
        }
    };

    // Cleanup concat file
    if !config.keep_concat && concat_file.exists() {
        fs::remove_file(&concat_file)?;
        println!("Cleaned up: {:?}", concat_file);
    }

    if success {
        Ok(())
    } else {
        Err("ffmpeg failed".into())
    }
}

fn find_ffmpeg() -> Result<String, Box<dyn std::error::Error>> {
    eprintln!("{}Searching for ffmpeg...{}", COLOR_YELLOW, COLOR_RESET);

    // Try to find in PATH using which/where
    let (cmd, args) = if cfg!(windows) {
        ("where", vec!["ffmpeg"])
    } else {
        ("which", vec!["ffmpeg"])
    };

    if let Ok(output) = Command::new(cmd).args(&args).output() {
        if output.status.success() {
            if let Ok(path) = String::from_utf8(output.stdout) {
                let path = path.lines().next().unwrap_or("").trim().to_string();
                if !path.is_empty() {
                    return Ok(path);
                }
            }
        }
    }

    // Deep search in common locations
    let (search_paths, ffmpeg_name): (Vec<PathBuf>, &str) = if cfg!(windows) {
        (vec![
            PathBuf::from("C:\\ffmpeg"),
            PathBuf::from("C:\\Program Files\\ffmpeg"),
            PathBuf::from("C:\\Program Files (x86)\\ffmpeg"),
            PathBuf::from(env::var("LOCALAPPDATA").unwrap_or_default())
                .join("Microsoft").join("WinGet").join("Packages"),
            PathBuf::from(env::var("USERPROFILE").unwrap_or_default())
                .join("scoop").join("apps").join("ffmpeg"),
            PathBuf::from(env::var("ChocolateyInstall").unwrap_or_else(|_| "C:\\ProgramData\\chocolatey".to_string()))
                .join("bin"),
            PathBuf::from("C:\\tools"),
        ], "ffmpeg.exe")
    } else {
        let home = env::var("HOME").unwrap_or_else(|_| ".".to_string());
        (vec![
            PathBuf::from("/usr/bin"),
            PathBuf::from("/usr/local/bin"),
            PathBuf::from("/opt"),
            PathBuf::from("/snap/bin"),
            PathBuf::from(&home).join(".local").join("bin"),
            PathBuf::from(&home).join("bin"),
            PathBuf::from("/opt/homebrew/bin"),
            PathBuf::from("/usr/share"),
            PathBuf::from("/Applications"),
        ], "ffmpeg")
    };

    eprintln!("{}Performing deep search in common directories...{}", COLOR_YELLOW, COLOR_RESET);

    for base in search_paths {
        if base.exists() {
            if let Some(found) = find_file_recursive(&base, ffmpeg_name) {
                return Ok(found.to_string_lossy().to_string());
            }
        }
    }

    Err("ffmpeg not found in PATH or common locations".into())
}

fn find_file_recursive(dir: &Path, name: &str) -> Option<PathBuf> {
    if let Ok(entries) = fs::read_dir(dir) {
        for entry in entries.flatten() {
            let path = entry.path();
            if path.is_file() && path.file_name().map(|n| n == name).unwrap_or(false) {
                return Some(path);
            }
            if path.is_dir() {
                if let Some(found) = find_file_recursive(&path, name) {
                    return Some(found);
                }
            }
        }
    }
    None
}

fn prompt_for_ffmpeg() -> Result<String, Box<dyn std::error::Error>> {
    println!("{}ffmpeg not found after comprehensive search{}", COLOR_RED, COLOR_RESET);
    let user_path = prompt_input("Enter path to ffmpeg executable (or press Enter to install): ");

    if !user_path.is_empty() {
        if Path::new(&user_path).exists() {
            println!("{}Using: {}{}", COLOR_GREEN, user_path, COLOR_RESET);
            return Ok(user_path);
        }
        return Err(format!("Not a valid executable: {}", user_path).into());
    }

    let reply = prompt_input("Install ffmpeg now? (y/n): ");
    if reply.to_lowercase() == "y" || reply.to_lowercase() == "yes" {
        return install_ffmpeg();
    }

    Err("ffmpeg required to continue".into())
}

fn install_ffmpeg() -> Result<String, Box<dyn std::error::Error>> {
    if cfg!(windows) {
        // Try winget
        if Command::new("where").arg("winget").output().map(|o| o.status.success()).unwrap_or(false) {
            println!("{}Installing ffmpeg via winget...{}", COLOR_YELLOW, COLOR_RESET);
            let _ = Command::new("winget")
                .args(&["install", "ffmpeg"])
                .status();
        } else {
            println!("Please download ffmpeg from https://ffmpeg.org/download.html");
            return Err("Cannot auto-install".into());
        }
    } else if cfg!(target_os = "macos") {
        if Command::new("which").arg("brew").output().map(|o| o.status.success()).unwrap_or(false) {
            println!("{}Installing ffmpeg via Homebrew...{}", COLOR_YELLOW, COLOR_RESET);
            Command::new("brew")
                .args(&["install", "ffmpeg"])
                .status()?;
        } else {
            return Err("Please install Homebrew first or download ffmpeg manually".into());
        }
    } else {
        // Linux
        if Command::new("which").arg("apt-get").output().map(|o| o.status.success()).unwrap_or(false) {
            let _ = Command::new("sudo").args(&["apt-get", "update"]).status();
            let _ = Command::new("sudo")
                .args(&["apt-get", "install", "-y", "ffmpeg"])
                .status();
        } else if Command::new("which").arg("yum").output().map(|o| o.status.success()).unwrap_or(false) {
            let _ = Command::new("sudo")
                .args(&["yum", "install", "-y", "ffmpeg"])
                .status();
        } else {
            return Err("Cannot auto-install. Please install manually".into());
        }
    }

    // Try to find it again
    find_ffmpeg().map_err(|_| "Installation failed".into())
}

fn find_video_files(dir: &Path) -> io::Result<Vec<PathBuf>> {
    let mut videos = Vec::new();

    for entry in fs::read_dir(dir)? {
        let entry = entry?;
        let path = entry.path();

        if path.is_file() {
            if let Some(ext) = path.extension() {
                if ext.to_string_lossy().to_lowercase() == "mp4" {
                    videos.push(path);
                }
            }
        }
    }

    Ok(videos)
}

fn create_concat_file(video_files: &[PathBuf], concat_file: &Path) -> io::Result<()> {
    let mut file = File::create(concat_file)?;

    for video in video_files {
        let abs_path = fs::canonicalize(video)?;
        let path_str = abs_path.to_string_lossy();

        // Escape single quotes
        let escaped_path = path_str.replace("'", "'\\''");
        writeln!(file, "file '{}'", escaped_path)?;
    }

    Ok(())
}

fn run_ffmpeg(ffmpeg_path: &str, concat_file: &Path, output_path: &Path) -> Result<(), Box<dyn std::error::Error>> {
    let status = Command::new(ffmpeg_path)
        .args(&[
            "-f", "concat",
            "-safe", "0",
            "-i", &concat_file.to_string_lossy(),
            "-c", "copy",
            &output_path.to_string_lossy(),
        ])
        .status()?;

    if !status.success() {
        return Err("ffmpeg command failed".into());
    }

    Ok(())
}
