#!/bin/bash
###############################################################################
# NVR Video Combiner - Bash Version
# Combines multiple MP4 video files using ffmpeg's concat demuxer (no re-encoding)
###############################################################################

set -euo pipefail

# Default values
DEFAULT_VIDEO_DIR="${HOME}/Videos/NVR"
OUTPUT_FILE="Combined_Output.mp4"
KEEP_CONCAT=false

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Usage information
usage() {
    cat << EOF
Usage: $0 [OPTIONS] [VIDEO_DIR]

Combine multiple MP4 video files using ffmpeg (no re-encoding)

Arguments:
    VIDEO_DIR           Directory containing MP4 files (default: ${DEFAULT_VIDEO_DIR})

Options:
    -o, --output FILE   Output filename (default: Combined_Output.mp4)
    --ffmpeg PATH       Path to ffmpeg executable (auto-detected if not specified)
    --keep-concat       Keep the concat.txt file after completion
    -h, --help          Show this help message

Examples:
    $0 /path/to/videos
    $0 -o MyVideo.mp4 /path/to/videos
    $0 --ffmpeg /usr/local/bin/ffmpeg /path/to/videos
EOF
    exit 0
}

# Find ffmpeg with comprehensive search
find_ffmpeg() {
    echo -e "${YELLOW}Searching for ffmpeg...${NC}" >&2
    
    # Check PATH first
    if command -v ffmpeg &> /dev/null; then
        command -v ffmpeg
        return 0
    fi

    # Deep search in common locations
    local search_paths=(
        "/usr/bin"
        "/usr/local/bin"
        "/opt"
        "/snap/bin"
        "${HOME}/.local/bin"
        "${HOME}/bin"
        "/opt/homebrew/bin"
        "/usr/share"
        "/Applications"
    )

    echo -e "${YELLOW}Performing deep search in common directories...${NC}" >&2
    for base in "${search_paths[@]}"; do
        if [[ -d "$base" ]]; then
            while IFS= read -r found; do
                if [[ -x "$found" ]]; then
                    echo "$found"
                    return 0
                fi
            done < <(find "$base" -name "ffmpeg" -type f 2>/dev/null)
        fi
    done

    return 1
}

# Parse command line arguments
VIDEO_DIR=""
FFMPEG_PATH=""

while [[ $# -gt 0 ]]; do
    case $1 in
        -h|--help)
            usage
            ;;
        -o|--output)
            OUTPUT_FILE="$2"
            shift 2
            ;;
        --ffmpeg)
            FFMPEG_PATH="$2"
            shift 2
            ;;
        --keep-concat)
            KEEP_CONCAT=true
            shift
            ;;
        -*)
            echo -e "${RED}Error: Unknown option: $1${NC}" >&2
            usage
            ;;
        *)
            VIDEO_DIR="$1"
            shift
            ;;
    esac
done

# Set default video directory if not specified
if [[ -z "$VIDEO_DIR" ]]; then
    read -p "Enter video directory path [${DEFAULT_VIDEO_DIR}]: " VIDEO_DIR
    VIDEO_DIR="${VIDEO_DIR:-$DEFAULT_VIDEO_DIR}"
fi

# Validate video directory
if [[ ! -d "$VIDEO_DIR" ]]; then
    echo -e "${RED}Error: Directory does not exist: $VIDEO_DIR${NC}" >&2
    exit 1
fi

# Find ffmpeg with comprehensive search and user prompts
if [[ -z "$FFMPEG_PATH" ]]; then
    if FFMPEG_PATH=$(find_ffmpeg); then
        echo -e "${GREEN}Found ffmpeg: $FFMPEG_PATH${NC}"
    else
        echo -e "${RED}ffmpeg not found after comprehensive search${NC}"
        read -p "Enter path to ffmpeg executable (or press Enter to install): " user_path
        
        if [[ -n "$user_path" ]]; then
            if [[ -x "$user_path" ]]; then
                FFMPEG_PATH="$user_path"
                echo -e "${GREEN}Using: $FFMPEG_PATH${NC}"
            else
                echo -e "${RED}Error: Not a valid executable: $user_path${NC}" >&2
                exit 1
            fi
        else
            read -p "Install ffmpeg now? (y/n): " -n 1 -r
            echo
            if [[ $REPLY =~ ^[Yy]$ ]]; then
                if command -v apt-get &> /dev/null; then
                    sudo apt-get update && sudo apt-get install -y ffmpeg
                elif command -v yum &> /dev/null; then
                    sudo yum install -y ffmpeg
                elif command -v brew &> /dev/null; then
                    brew install ffmpeg
                else
                    echo -e "${RED}Error: Cannot auto-install. Please install manually.${NC}" >&2
                    exit 1
                fi
                FFMPEG_PATH=$(find_ffmpeg) || {
                    echo -e "${RED}Error: Installation failed${NC}" >&2
                    exit 1
                }
            else
                echo -e "${RED}ffmpeg required to continue${NC}" >&2
                exit 1
            fi
        fi
    fi
fi

if [[ ! -x "$FFMPEG_PATH" ]]; then
    echo -e "${RED}Error: ffmpeg not executable: $FFMPEG_PATH${NC}" >&2
    exit 1
fi

# Setup paths
CONCAT_FILE="${VIDEO_DIR}/concat.txt"
OUTPUT_PATH="${VIDEO_DIR}/${OUTPUT_FILE}"

# Find all MP4 files
mapfile -t VIDEO_FILES < <(find "$VIDEO_DIR" -maxdepth 1 -name "*.mp4" -type f | sort)

if [[ ${#VIDEO_FILES[@]} -eq 0 ]]; then
    echo -e "${RED}Error: No .mp4 files found in $VIDEO_DIR${NC}" >&2
    exit 1
fi

# Display found videos
echo -e "${GREEN}Found ${#VIDEO_FILES[@]} video file(s):${NC}"
for video in "${VIDEO_FILES[@]}"; do
    echo "  - $(basename "$video")"
done

# Create concat file
echo -e "\n${YELLOW}Creating concat file: $CONCAT_FILE${NC}"
> "$CONCAT_FILE"  # Clear file

for video in "${VIDEO_FILES[@]}"; do
    # Escape single quotes in path
    escaped_path="${video//\'/\'\\\'\'}"
    echo "file '$escaped_path'" >> "$CONCAT_FILE"
done

# Run ffmpeg
echo -e "\n${YELLOW}Running ffmpeg...${NC}"
echo -e "Output will be: ${GREEN}$OUTPUT_PATH${NC}\n"

if "$FFMPEG_PATH" -f concat -safe 0 -i "$CONCAT_FILE" -c copy "$OUTPUT_PATH"; then
    echo -e "\n${GREEN}✓ Successfully created: $OUTPUT_PATH${NC}"
    SUCCESS=true
else
    echo -e "\n${RED}✗ Error running ffmpeg${NC}" >&2
    SUCCESS=false
fi

# Cleanup concat file
if [[ "$KEEP_CONCAT" == false ]] && [[ -f "$CONCAT_FILE" ]]; then
    rm "$CONCAT_FILE"
    echo -e "Cleaned up: $CONCAT_FILE"
fi

[[ "$SUCCESS" == true ]] && exit 0 || exit 1